<?php

namespace App\Http\Controllers\Admin;

use App\Helper\AppConstant;
use App\Http\Controllers\Controller;
use App\Jobs\SendEmailPeriod;
use App\Models\Period;
use App\Models\User;
use App\Models\UserPeriod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

use App\Mail\RemindPeriod;
use App\Mail\AssignPeriod;

use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class PeriodController extends Controller
{
    public function periodList(){
        $period = new Period();
        $list = $period->getListForAdmin();
        $listPeriodType = AppConstant::getListPeriodType();

        if( isset($_GET['debug']) ){
            $periodId  = 1;
            $period    = Period::where('id', $periodId)->firstOrFail();
            $email     = 'info@oavaangifte.nl';
            $user      = User::where('email',$email)->first();
            Mail::to($email)->send(new RemindPeriod($period, $user));
        }

        $params = [
            'list' => $list,
            'listPeriodType' => $listPeriodType,
        ];

        return view('admin.period.index', $params);
    }

    public function getListUserForAssignToPeriod(Request $request){
        $periodId = $request->get('period_id');
        // get all users without declaration
        $userModel = new User();
        $listUsers = $userModel->getUserNotDeclaration($periodId);
        return response()->json($listUsers);
    }

    public function ajaxGetUserNotDeclatationOfPeriod(Request $request){
        $periodId = $request->get('period_id');
        // get all users without declaration for this period
        $userPeriod = new UserPeriod();
        $listUsers = $userPeriod->getListUnfilledOfPeriod($periodId);
        return response()->json($listUsers);
    }

    public function updatePeriod($id){
        $period = Period::where('id', $id)->firstOrFail();

        $params = [
            'period' => $period
        ];

        return view('admin.period.update', $params);
    }

    public function savePeriod(Request $request){
        $data = $request->all();
        $validator = Validator::make($data, [
            'period' => 'required',
            'description' => 'required',
            'id' => 'required',
            'date_statement' => 'required',
            'date_release' => 'required'
        ]);

        if(empty($data['id'])){
            abort(404);
        }

        if($validator->fails()){
            return redirect(route('', ['id' => $data['id']]))
                ->withErrors($validator->errors())
                ->withInput();
        } else {
            $period = Period::where('id', $data['id'])->firstOrFail();
            $period->period = $data['period'];
            $period->description = $data['description'];
            $period->date_statement = \DateTime::createFromFormat('d-m-Y', $data['date_statement']);
            $period->date_release = \DateTime::createFromFormat('d-m-Y', $data['date_release']);
            $period->save();

            return redirect(route('admin.period.periodList'))->with(['b_a_message' => "Saved!"]);
        }
    }

    public function emailPeriod($id){
        $period = Period::where('id', $id)->firstOrFail();
        $userPeriod = new UserPeriod();
        $list = null;
        if($period->type  == 3){
            $list = $userPeriod->getListFilledForQ6($id);
        }else{
            $list = $userPeriod->getListUnfilledOfPeriod($id);
        }
        $params = [
            'list' => $list,
            'period' => $period,
        ];

        return view('admin.period.send_mail', $params);
    }

    public function sendPeriod(Request $request){
        $data = $request->all();
        $validator = Validator::make($data, [
            'id'   => 'required',
            'mode' => 'required'
        ]);
        if($validator->fails()){
            return redirect(route(''))
                ->withErrors($validator->errors())
                ->withInput();
        } else {
            $periodId = $data['id'];
            $userId   = $data["user_id"] ?? 0;

            $period = Period::where('id', $periodId)->firstOrFail();

            // Frontend sends comma-separated strings; convert to arrays
            $emails     = !empty($data['email']) ? $data['email'] : '';
            $emails     = is_array($emails) ? $emails : explode(',', $emails);
            $emails     = array_map(function($e){ return trim((string)$e); }, $emails);

            $arrayUserId = [];
            if(is_string($userId)){
                $arrayUserId = array_map('trim', explode(",", $userId));
            } elseif (is_array($userId)) {
                $arrayUserId = $userId;
            }

            // Save users to period
            $modelUserPeriod = new UserPeriod();
            foreach($arrayUserId as $_user_id){
                if($_user_id === '' || $_user_id === null) continue;
                $params = [
                    "user_id"   => $_user_id,
                    "period_id" => $periodId
                ];
                $modelUserPeriod->updateUser($params);
            }

            // Build recipients (parallel index arrays -> structured list)
            $recipients = $this->buildRecipients($emails, $arrayUserId);

            //Log::info('DEBUG: sendPeriod -> recipients built', [...]);

            // type_email = 1 => modal "assign user" sends per-user
            $this->doSendEmailPeriod($recipients, $period, $data['mode'], 1);

            return redirect(route('admin.period.periodList'))->with(['b_a_message' => "System started sending emails!"]);
        }
    }

    // Build recipients list from parallel arrays
    private function buildRecipients(array $emails, array $userIds): array
    {
        $max = max(count($emails), count($userIds));
        $out = [];
        for ($i = 0; $i < $max; $i++) {
            $email  = $emails[$i]  ?? null;
            $userId = $userIds[$i] ?? null;

            if (is_string($email)) {
                $email = trim(preg_split('/[;,]/', $email)[0]);
                if ($email === '') { $email = null; }
            }

            $out[] = [
                'user_id' => $userId !== '' ? $userId : null,
                'email'   => $email,
            ];
        }
        return $out;
    }

    // Send mails for a period
    private function doSendEmailPeriod($payload, Period $period, $type, $type_email = 0)
    {
        //Log::info('DEBUG: Entered doSendEmailPeriod', [...]);

        $this->type         = $type;
        $this->period       = $period;
        $this->_type_email  = $type_email;

        /**
         * type_email == 0
         * = legacy flow (reminder mails), leave mostly as-is.
         */
        if ($this->_type_email == 0) {
            $listEmail = [];
            if($this->type == 1){
                $list = User::whereIn('id', $payload)->get();
            } elseif($this->type == 3){
                $list = User::whereNotIn('id', $payload)->get();
            } else if($this->type == 4){
                $listEmail = is_array($payload) ? $payload : [];
            } elseif($this->type == 5){ 
                // All users who did NOT submit declaration
                $userModel = new User();
                $listUserNotSubmitDeclaration = $userModel->getUserNotDeclaration($this->period->id);
                $listEmail = $listUserNotSubmitDeclaration->pluck('email')->toArray();
            } else{
                $list = User::all();
            }

            if(!empty($listEmail)){
                foreach ($listEmail as $email){
                    $email = is_string($email) ? trim($email) : '';
                    if($email !== '' && filter_var($email, FILTER_VALIDATE_EMAIL)){
                        $user = User::where('email',$email)->first();
                        Mail::to($email)->send(new RemindPeriod($this->period, $user));
                        //Log::info('DEBUG: Remind mail sent', [...]);
                    } else {
                        //Log::warning('DEBUG: Skipped invalid email', [...]);
                    }
                }
            } else {
                foreach ($list ?? [] as $user){
                    $email = trim((string)$user->email);
                    if($email !== '' && filter_var($email, FILTER_VALIDATE_EMAIL)){
                        Mail::to($email)->send(new RemindPeriod($this->period, $user));
                        //Log::info('DEBUG: Remind mail sent', [...]);
                    } else {
                        //Log::warning('DEBUG: Skipped invalid user email', [...]);
                    }
                }
            }

            return;
        }

        /**
         * type_email == 1
         * = new flow: assign user to period + send mail
         */
        if ($this->_type_email == 1) {

            if (!is_array($payload)) {
                //Log::warning('DEBUG: Payload not array');
                return;
            }

            foreach ($payload as $idx => $rec) {
                $rawEmail = isset($rec['email']) ? trim((string)$rec['email']) : null;
                $userId   = isset($rec['user_id']) ? trim((string)$rec['user_id']) : null;

                //Log::info('DEBUG: Iterating recipient', [...]);

                $user = null;
                if ($userId !== null && $userId !== '') {
                    $user = User::find($userId);
                }
                if (!$user && $rawEmail) {
                    $user = User::where('email', $rawEmail)->first();
                }

                if (!$user) {
                    //Log::warning('DEBUG: No matching user', [...]);
                    continue;
                }

                $to = $rawEmail ?: $user->email;
                $to = trim((string)$to);

                if ($to === '' || !filter_var($to, FILTER_VALIDATE_EMAIL)) {
                    //Log::warning('DEBUG: Invalid recipient email', [...]);
                    continue;
                }

                try {
                    Mail::to($to)->send(new AssignPeriod($this->period, $user));
                    //Log::info('DEBUG: AssignPeriod mail sent', [...]);
                } catch (\Exception $e) {
                    //Log::error('DEBUG: AssignPeriod mail failed', [...]);
                }
            }
        }
    }

    public function sendRemindPeriod(Request $request){
        $data = $request->all();
        $validator = Validator::make($data, [
            'id' => 'required',
            'mode' => 'required'
        ]);
        if($validator->fails()){
            return redirect(route(''))
                ->withErrors($validator->errors())
                ->withInput();
        } else {
            $periodId = $data['id'];
            $userId = 0;
            if(isset($data["user_id"])){
                $userId = $data["user_id"];
            }
            $period = Period::where('id', $periodId)->firstOrFail();
            $emails = !empty($data['email']) ? $data['email'] : [0];
            $arrayUserId = [];
            if(is_string($emails)){
                $emails = explode(',', $emails);
            }
            if(is_string($userId)){
                $arrayUserId = explode(",",$userId);
            }
            // Keep job-flow unchanged
            $sendEmail = SendEmailPeriod::dispatch($emails, $period, $data['mode'],0);

            // Save users to period
            $modelUserPeriod = new UserPeriod();
            foreach($arrayUserId as $_user_id){
                $params = array(
                    "user_id" => $_user_id,
                    "period_id" => $periodId
                );
                $modelUserPeriod->updateUser($params);
            }
            return redirect(route('admin.period.periodList'))->with(['b_a_message' => "System started sending emails!"]);
        }
    }

    // POST assign users to period
    public function assignUser(Request $request){
        $params = $request->all();
        if($params['user_id'] && $params['period_id']){
            $userPeriodModel = new UserPeriod();
            $result = $userPeriodModel->updateUser($params);
            if($result){
                return redirect(route('admin.period.emailPeriod',['id' => $params['period_id']]))->with(['b_a_message' => "Saved!"]);
            }
        }
        return redirect(route('admin.period.emailPeriod',['id' => $params['period_id']]));
    }
}
