<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\RecyclingContribution;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;
use App\Models\Menu;
use PhpOffice\PhpWord\TemplateProcessor;
use PhpOffice\PhpWord\Shared\Converter;
use Illuminate\Support\Facades\Mail;
use App\Mail\RecyclingContributionMail;
use App\Helper\AppConstant;

class RecyclingContributionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        View::share('menu', Menu::all());
    }
    // Frontend
    public function create()
    {
        $user           = Auth::user();
        $templatePath   = public_path('storage/recycling_contributions/template-2026.docx');
        $outputPath     = public_path('storage/recycling_contributions/downloads/recycling-contributions-'.$user->id.'.docx');
        $outputUrl      = asset('storage/recycling_contributions/downloads/recycling-contributions-'.$user->id.'.docx'.'?t='.time());
        if (!file_exists($outputPath) ) {
            $template = new TemplateProcessor($templatePath);
            $template->setValue('company_name', htmlspecialchars($user->company_name));
			$template->setValue('address', htmlspecialchars($user->address));
			$template->setValue('postcode', htmlspecialchars($user->postcode));
			$template->setValue('place', htmlspecialchars($user->place));
            $template->setValue('created_at', date('d-m-Y'));
            if (!file_exists(public_path('storage/recycling_contributions/downloads'))) {
                mkdir(public_path('storage/recycling_contributions/downloads'), 0777, true);
            }
            $template->saveAs($outputPath);
        }
        $params = [
            'user'          => $user,
            'outputPath'    => $outputPath,
            'outputUrl'     => $outputUrl,
        ];
        return view('front.recycling_contributions.create',$params);
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        $validated = $request->validate([
            'contact_persoon' => 'required|string|max:255',
        ], [
            'contact_persoon.required' => 'Contactpersoon is verplicht.',
        ]);
        $validated['user_id'] = $user->id;

        //signature
        $data = $request->signature;
        $image = $data[1];
        $image = base64_decode($image);
        $path = 'storage/recycling_contributions/downloads/recycling-contributions-signatures-' . $user->id . '.png';
        $fullPath = public_path($path);
        file_put_contents($fullPath, $image);
        $validated['signature'] = $path;

        // Insert sign
        $outputPath     = public_path('storage/recycling_contributions/downloads/recycling-contributions-'.$user->id.'.docx');
        $tempPath       = public_path('storage/recycling_contributions/downloads/temp-recycling-contributions-'.$user->id.'.docx');
        $template = new TemplateProcessor($outputPath);
        // $template->setValue('contact_persoon', $validated['contact_persoon']);
		$contactPersoon = htmlspecialchars($validated['contact_persoon'], ENT_QUOTES, 'UTF-8');
        $template->setValue('contact_persoon', "<w:r><w:rPr><w:color w:val='000000'/></w:rPr><w:t>{$contactPersoon}</w:t></w:r>");

        $template->setValue('created_at', date('d-m-Y'));
        $template->setImageValue('signature', [
            'path' => $fullPath,
            'width' => Converter::cmToPixel(14),
            'height' => Converter::cmToPixel(12), 
            'ratio' => true,
        ]);
        $template->saveAs($tempPath);
        rename($tempPath, $outputPath);

        RecyclingContribution::updateOrCreate(
            ['user_id' => $user->id],
            $validated 
        );

        Mail::to( config('mail.admin_email') )->send(new RecyclingContributionMail($user, $fullPath));

        return response()->json([
            'success' => true,
            'message' => 'Bedrijf succesvol toegevoegd!',
            'url' => asset($path)
        ]);
    }
    public function download($user_id = 0){

        $user = Auth::user();
        $user_id = $user_id ? $user_id : $user->id;
        $item = RecyclingContribution::where('user_id',$user_id)->first();
        if( !$item ){
            return redirect()->route('recycling_contributions.create');
        }
        $outputPath = public_path('storage/recycling_contributions/downloads/recycling-contributions-'.$user_id.'.docx');
        if (file_exists($outputPath)) {
            return response()->download($outputPath);
        }else{
            return redirect()->route('home.index');
        }
    }
}

